/*********************************************************
** Copyright (c) 2005
** University of Washington
** Licensed under the terms set forth by University of
** Washington. If you did not sign such a license, you
** are using this software/code illegally and you do not
** have permission to use, modify, or redistribute
** this or any files in this software package.
**
**
** File : Cluster.h
** Header file for clustering algorithm classes.
** The main class is CCluster. Other classes encapsulate
** distance functions and data-combination functions.
**********************************************************/

#ifndef CLUSTER_H
#define CLUSTER_H

#include "ClusterNode.h"
#include "Matrix.h"
#include "ProgressThread.h"
#include "IProgressThreadCallback.h"
#include <stdint.h>

/* 
 * Measures the similarity between two data points. 
 * Abstract base class - must be inherited.
 */
class CDistanceFunction
{
 public:
  virtual ~CDistanceFunction() {}
  virtual double Distance(CClusterNode* pData1, CClusterNode* pData2) = 0;
  virtual uintmax_t GetMemoryRequirement() = 0;
  
};

/*
 * Combines two rows of data.
 * Abstract base class - must be inherited.
 */
class CCombineDataFunction
{
 public:
  virtual ~CCombineDataFunction() {}
  virtual void* CombineData(CClusterNode* pNode1, CClusterNode* pNode2) = 0;
  virtual uintmax_t GetMemoryRequirement() = 0;

};

/* 
 * Encapsulates a clustering algorithm. 
 * Abstract base class - must be inherited.
 */
class CCluster : public IProgressThreadCallback
{
 public:
  CCluster(CDistanceFunction* pFunc, CCombineDataFunction *pCombine);
  virtual ~CCluster() {}

  CDistanceFunction* GetDistanceFunction();
  CCombineDataFunction* GetCombineDataFunction();
  virtual CMatrix* GetDistanceMatrix() = 0;

  virtual CClusterNode* Cluster(void **ppData, int iDataCount) = 0;
  /* Returns number of bytes required to cluster data with given size & count */
  virtual uintmax_t GetMemoryRequirement(uintmax_t iDataSize, uintmax_t iDataCount) = 0;
  virtual void CheckMemory(uintmax_t iDataSize, uintmax_t iDataCount);
  
 private:
  CDistanceFunction* m_pDistance;
  CCombineDataFunction* m_pCombineData;
  int foo;
};

#endif
