/*********************************************************
** Copyright (c) 2005
** University of Washington
** Licensed under the terms set forth by University of
** Washington. If you did not sign such a license, you
** are using this software/code illegally and you do not
** have permission to use, modify, or redistribute
** this or any files in this software package.
**
** File: edge.c  $Revision: 254 $
**
**********************************************************/
#include <assert.h>
#include <stdio.h>
#include "edge.h"
#include "ProgressThreadDef.h"
#include "MsgIds.h"
#include "debug.h"

// Exp[-0.025 i]
static double exptab[]={
   1,0.9753099120283326,0.951229424500714,0.9277434863285529,
    0.9048374180359595,0.8824969025845955,0.8607079764250578,
    0.8394570207692074,0.8187307530779818,0.7985162187593771,
    0.7788007830714049,0.7595721232249685,0.7408182206817179,
    0.7225273536420722,0.7046880897187134,0.6872892787909722,
    0.6703200460356393,0.6537697851298473,0.6376281516217733,0.62188505646502,
    0.6065306597126334,0.5915553643668151,0.5769498103804866,
    0.5627048688069557,0.5488116360940264,0.5352614285189903,
    0.522045776761016,0.5091564206075492,0.49658530379140947,
    0.4843245689553624,0.4723665527410147,0.4607037809989658,
    0.44932896411722156,0.4382349924649492,0.42741493194872665,
    0.4168620196785084,0.4065696597405991,0.39653141907499284,
    0.3867410234545012,0.3771923535631569,0.36787944117144233,
    0.35879646540595156,0.3499377491111553,0.3412977553009937,
    0.33287108369807955,0.32465246735834974,0.31663676937905316,
    0.30881897968801986,0.301194211912202,0.29375770032353277,
    0.2865047968601901,0.2794309682214073,0.2725317930340126,
    0.26580295908892654,0.2592402606458915,0.25283959580474646,
    0.24659696394160643,0.24050846320834213,0.23457028809379762,
    0.22877872704522242,0.22313016014842982,0.21762105686523284,
    0.21224797382674304,0.2070075526811526,0.20189651799465538,
    0.19691167520419406,0.19204990862075408,0.18730817948195702,
    0.1826835240527346,0.1781730517728984,0.17377394345044514,
    0.16948344949947006,0.16529888822158653,0.16121764412977674,
    0.1572371663136276,0.15335496684492847,0.14956861922263504,
    0.14587575685622736,0.14227407158651353,0.13876131224295524,
    0.1353352832366127,0.13199384318783022,0.12873490358780418,
    0.1255564274931972,0.1224564282529819,0.11943296826671962,
    0.11648415777349697,0.11360815367076371,0.11080315836233387,
    0.10806741863482926,0.10539922456186433,0.1027969084352864,
    0.1002588437228037,0.09778344405135005,0.09536916221554961,
    0.09301448921066349,0.09071795328941247,0.08847811904208727,
    0.0862935864993705,0.08416299025731036,0.0820849986238988,
    0.08005831278672051,0.07808166600115313,0.07615382279861033,
    0.07427357821433388,0.07243975703425146,0.07065121306042957,
    0.06890682839466256,0.06720551273974976,0.06554620271802433,
    0.06392786120670757,0.0623494766896734,0.06081006262521795,
    0.05930865682943872,0.057844320874838456,0.05641613950377735,
    0.05502322005640721,0.05366469191273011,0.05233970594843238,
    0.051047434004154395,0.049787068367863944,0.048557821270009946,
    0.04735892439114091,0.0461896283816801,0.0450492023935578,
    0.04393693362340742,0.042852126867040166,0.041794104084919896,
    0.04076220397836621,0.039755781576221304,0.03877420783172201,
    0.03781686922932507,0.036883167401239994,0.035972518753429654,
    0.035084354100845025,0.03421811831166603,0.033373269960326066,
    0.03254928098910342,0.03174563637806794,0.030961833823176882,
    0.0301973834223185,0.02945180736910728,0.028724639654239423,
    0.028015425774221808,0.02732372244729256,0.026649097336355485,
    0.025991128778755333,0.02534940552272493,0.024723526470339388,
    0.024113100426816865,0.023517745856009107,0.02293709064192892,
    0.02237077185616559,0.021818435531042762,0.02127973643837717,
    0.020754337873699742,0.02024191144580438,0.019742136871492774,
    0.01925470177538692,0.0187793014946844,0.01831563888873418,
    0.017863424153314034,0.017422374639493515,0.016992214676969066,
    0.016572675401761237,0.016163494588165874,0.015764416484854486,
    0.015375191655021433,0.014995576820477703,0.014625334709594208,
    0.014264233908999256,0.01391204871893762,0.013568559012200934,
    0.013233550096540928,0.012906812580479862,0.012588142242433998,
    0.012277339903068436,0.011974211300803622,0.011678566970395442,
    0.011390222124513367,0.011108996538242306,0.010834714436436152,
    0.010567204383852655,0.01030629917800074,0.010051835744633576,
    0.009803655035821828,0.009561601930543505,0.009325525137728326,
    0.009095277101695816,0.008870713909928251,0.008651695203120634,
    0.00843808408745153,0.008229747049020023,0.008026553870395154,
    0.007828377549225767,0.007635094218859962,0.007446583070924338,
    0.0072627262798148845,0.0070834089290521185,0.006908518939454525,
    0.006737946999085467,0.006571586494929613,0.006409333446256377,
    0.006251086439628561,0.006096746565515633,0.005946217356472094,
    0.005799404726842141,0.005656216913953104,0.0055165644207607716,
    0.005380359959910816,0.005247518399181385,0.005117956708272651,
    0.004991593906910213,0.004868351014229763,0.004748150999411473,
    0.004630918733533246,0.004516580942612666,0.004405066161808402,
    0.00429630469075234,0.004190228549984576,0.004086771438464067,
    0.0039858686921282905,0.003887457243476127,0.003791475582148608,
    0.003697863716482929,0.0036065631360157305,0.0035175167749121284,
    0.0034306689762977315,0.003345965457471272,0.0032633532759761452,
    0.003182780796509667,0.0031041976586493087,0.0030275547453758127,
    0.0029528041523734466,0.0028798991580882404,0.0028087941945255128,
    0.0027394448187683684,0.002671807685199448,0.0026058405184084983,
    0.002541502086768856,0.0024787521766663585,0.0024175515673645035,
    0.0023578620064902307,0.002299646186124936,0.002242867719485801,
    0.002187491118182885,0.002133481770037708,0.0020808059174495276,
    0.002029430636295734,0.001979323815353195,0.0019304541362277093,
    0.0018827910537789773,0.0018363047770289056,0.0017909662505412698,
    0.0017467471362611182,0.001703619795802574,0.001661557273173934,
    0.0016205332779293052,0.001580522168736217,0.001541498937348949,
    0.0015034391929775724,0.001466319147042903,0.0014301155983078731,
    0.0013948059183759988,0.0013603680375478928,0.0013267804310269915,
    0.001294022105465848,0.0012620725858446134,0.001230911902673481,
    0.0012005205795111,0.0011708796207911744,0.0011419704999496075,
    0.0011137751478448024,0.0010862759414638579,0.001059455692907609,
    0.001033297638647637,0.0010077854290485105,0.0009829031181487378,
    0.0009586351536940199,0.0009349663674165814,0.0009118819655545162,
    0.000889367519605198,0.0008674089573070017,0.0008459925538436801,
    0.0008251049232659038,0.0008047330101246132,0.0007848640813109316,
    0.0007654857180975626,0.0007465858083766792,0.0007281525390894604,
    0.000710174388842549,0.0006926401207068012,0.0006755387751938437,
    0.0006588596634060358,0.0006425923603555573,0.0006267266984484576,
    0.0006112527611295723,0.0005961608766843584,0.0005814416121937556,
    0.0005670857676383035,0.0005530843701478336,0.0005394286683931291,
    0.0005261101271160632,0.0005131204217947829,0.0005004514334406104,
    0.000488095243523415,0.00047604412902226933,0.00046429055759831357,
    0.00045282718288679695,0.00044164683990535953,0.00043074254057568753,
    0.00042010746935573415,0.00040973497897978643,0.00039961858630370645,
    0.0003897519682527545,0.0003801289578694637,0.0003707435404590882,
    0.00036158984983022583,0.00035266216462825575,0.0003439549047593054,
    0.00033546262790251185,0.00032718002610839203,0.00031910192248120326,
    0.00031122326794321407,0.0003035391380788668,0.0002960447300568554,
    0.000288735359628203,0.00028160645819845157,0.00027465356997214205,
    0.00026787234916779776,0.00026125855730166754,0.0002548080605385384,
    0.00024851682710795185,0.00024238092478421678,0.00023639651842864072,
    0.00023055986759244163,0.0002248673241788482,0.0002193153301629189,
    0.0002139004153676611,0.00020861919529505766,0.00020346836901064417,
    0.00019844471708031962,0.00019354509955809383,0.00018876645402351932,
    0.0001841057936675792,0.00017956020542583297,0.00017512684815765842,
    0.00017080295087044495,0.00016658581098763324,0.00016247279265951725,
    0.00015846132511575126,0.0001545489010585363,0.0001507330750954765,
    0.00014701146221112919,0.00014338173627629318,0.00013984162859410104,
    0.0001363889264820114,0.0001330214718888092,0.0001297371600457538,
    0.00012653393815103005,0.00012340980408667956,0.00012036280516721316,
    0.00011739103691911796,0.00011449264189049966,0.00011166580849011478,
    0.00010890876985506645,0.00010621980274645875,0.00010359722647231549,
    0.00010103940183709324,0.00009854473011713091,0.0000961116520613947,
    0.00009373864691689655,0.00009142423147817327,0.00008916695916023506,
    0.00008696541909440292,0.00008481823524646916,0.00008272406555663223,
    0.00008068160110066498,0.00007868956527179456,0.00007674671298278181,
    0.0000748518298877006,0.00007300373162293296,0.00007120126306690273,
    0.00006944329761808704,0.00006772873649085378,0.00006605650802868481,
    0.00006442556703435542,0.00006283489411665261,0.00006128349505322202,
    0.00005977040016914672,0.00005829466373086881,0.00005685536335507487,
    0.000055451599432176945,0.00005408249456402682,0.00005274719301551376,
    0.00005144486017970228,0.00005017468205617528,0.00004893586474225786,
    0.00004772763393680188,0.00004654923445622269,0.000045399929762484854,
    0.00004427900150274157,0.000043185749060341275,0.00004211948911691908,
    0.00004107955522530065,0.00004006529739295107,0.00003907608167570808,
    0.000038111289781546776,0.000037170318684126666,0.00003625258024588065,
    0.00003535750085040998,0.00003448452104395504,0.00003363309518571897,
    0.000032802691106824094,0.000031992789777689165,0.000031202884983619004,
    0.000030432483008403625,0.00002968110232572986,0.000028948273298211517,
    0.000028233537883850797,0.000027536449349747158,0.000026856571992874528,
    0.00002619348086775303,0.000025546761520844016,0.00002491600973150316,
    0.00002430083125932946,0.000023700841597751968,0.00002311566573370091,
    0.00002254493791321217,0.000021988301412819176,0.000021445408316589164,
    0.000020915919298664244,0.000020399503411171922,0.00001989583787737175,
    0.000019404607889909406,0.00001892550641405186,0.000018458233995780558,
    0.00001800249857462311,0.00001755801530110584,0.000017124506358713654,
    0.00001670170079024566,0.00001628933432845802,0.00001588714923088849,
    0.000015494894118758837,0.000015112323819855006,0.000014739199215286483,
    0.000014375287090029125,0.000014020359987158328,0.000013674196065680938,
    0.000013336578961877444,0.00001300729765406762,0.000012686146330715025,
    0.000012372924261788221,0.000012067435673297889,0.000011769489624931723,
    0.00001147889989071055,0.00001119548484259094,0.000010919067336941897,
    0.000010649474603824239,0.000010386538139003777,0.00001013009359863071,
    9.879980696519291e-6,9.636043103963851e-6,9.398128352028201e-6,
    9.166087736247602e-6,8.93977622368364e-6,8.719052362273868e-6,
    8.50377819241975e-6,8.293819160757357e-6,8.089044036057154e-6,
    7.889324827200223e-6,7.694536703179588e-6,7.504557915076858e-6,
    7.319269719965135e-6,7.138556306690833e-6,6.962304723487946e-6,
    6.79040480737947e-6,6.6227491153220365e-6,6.459232857050451e-6,
    6.299753829580389e-6,6.14421235332821e-6,5.992511209807929e-6,
    5.8445555808665665e-6,5.70025298941967e-6,5.559513241650137e-6,
    5.422248370634153e-6,5.288372581358964e-6,5.157802197098256e-6,
    5.030455607111439e-6,4.906253215634289e-6,4.785117392129009e-6,
    4.666972422762587e-6,4.551744463083231e-6,4.439361491865156e-6,
    4.3297532660929705e-6,4.222851277057527e-6,4.118588707535708e-6,
    4.016900390027534e-6,3.917722766024328e-6,3.820993846282582e-6,
    3.726653172078671e-6,3.6346417774201543e-6,3.544902152190152e-6,
    3.457378206201623e-6,3.3720152341391786e-6,3.2887598813664843e-6,
    3.2075601105778547e-6,3.128365169273275e-6,3.051125558036417e-6,
    2.975792999595894e-6,2.902320408650404e-6,2.830661862438859e-6,
    2.7607725720371986e-6,2.6926088543638333e-6,2.626128104876299e-6,
    2.5612887709420388e-6,2.498050325866635e-6,2.4363732435633346e-6,
    2.3762189738479388e-6,2.3175499183436873e-6,2.2603294069810542e-6,
    2.2045216750777447e-6,2.150091840984627e-6,2.0970058842835516e-6,
    2.045230624523486e-6,1.994733700481656e-6,1.9454835499367136e-6,
    1.8974493899413437e-6,1.8506011975819048e-6,1.804909691213134e-6,
    1.7603463121561693e-6,1.7168832068484328e-6,1.674493209434266e-6,
    1.6331498247853738e-6,1.5928272119405093e-6,1.5535001679540347e-6,
    1.515144112143249e-6,1.477735070724678e-6,1.4412496618296672e-6,
    1.4056650808899562e-6,1.3709590863840845e-6,1.3371099859357042e-6,
    1.3040966227551563e-6,1.2718983624157768e-6,1.2404950799567113e-6,
    1.2098671473041613e-6,1.179995421003191e-6,1.150861230252457e-6,
    1.1224463652343422e-6,1.0947330657332276e-6,1.0677040100347827e-6,
    1.0413423040993216e-6,1.0156314710024903e-6,9.905554406366444e-7,
    9.660985396665116e-7,9.422454817328475e-7,9.189813578979571e-7,
    8.962916273271339e-7,8.741621082001578e-7,8.525789688471974e-7,
    8.315287191035679e-7,8.109982018779327e-7,7.909745849287022e-7,
    7.714453528434592e-7,7.523982992164199e-7,7.338215190190347e-7,
    7.15703401158952e-7,6.980326212227157e-7,6.807981343976331e-7,
    6.639891685684082e-7,6.475952175842209e-7,6.316060346920352e-7,
    6.160116261320527e-7,6.008022448912823e-7,5.859683846113411e-7,
    5.715007736466721e-7,5.573903692694596e-7,5.436283520176363e-7,
    5.302061201824281e-7,5.171152844320074e-7,5.04347662567888e-7,
    4.91895274410782e-7,4.797503368127321e-7,4.679052587923886e-7,
    4.563526367903986e-7,4.450852500419419e-7,4.3409605606351465e-7,
    4.2337818625115253e-7,4.1292494158732645e-7,4.027297884538396e-7,
    3.927863545481039e-7,3.8308842490024056e-7,3.73629937988526e-7,
    3.6440498195074056e-7,3.5540779088906284e-7,3.4663274126619643e-7,
    3.3807434839047367e-7,3.297272629877487e-7,3.2158626785792396e-7,
    3.1364627461403154e-7,3.059023205018258e-7,2.9834956529789845e-7,
    2.9098328828438453e-7,2.837988852983579e-7,2.7679186585408023e-7,
    2.699578503363014e-7,2.6329256726285584e-7,2.5679185061484973e-7,
    2.504516372327617e-7,2.4426796427683665e-7,2.382369667501818e-7,
    2.3235487508301652e-7,2.26618012776571e-7,2.2102279410515295e-7,
    2.1556572187495293e-7,2.1024338523818466e-7,2.0505245756119267e-7,
    1.9998969434520019e-7,1.9505193119839026e-7,1.9023608185805834e-7,
    1.8553913626159784e-7,1.8095815866511173e-7,1.7649028580847912e-7,
    1.7213272512572301e-7,1.6788275299956603e-7,1.6373771305908126e-7,
    1.5969501451937286e-7,1.5575213056225282e-7,1.5190659675689612e-7,
    1.481560095194917e-7,1.4449802461092448e-7,1.4093035567154855e-7,
    1.374507727921396e-7,1.3405710112012798e-7,1.3074721950024527e-7,
    1.2751905914873347e-7,1.2437060236028695e-7,1.2129988124692217e-7,
    1.1830497650798283e-7,1.1538401623051466e-7,1.1253517471925912e-7,
    1.0975667135553342e-7,1.0704676948428807e-7,1.0440377532863836e-7,
    1.0182603693119986e-7,9.931194312156244e-8,9.685992250925363e-8,
    9.44684425015714e-8,9.213600834566135e-8,8.98611621942485e-8,
    8.764248219443636e-8,8.547858159900026e-8,8.336810789962771e-8,
    8.130974198155457e-8,7.930219729907625e-8,7.734421907141565e-8,
    7.543458349844232e-8,7.35720969957598e-8,7.175559544867455e-8,
    6.998394348458725e-8,6.82560337633487e-8,6.657078628513438e-8,
    6.492714771541144e-8,6.332409072656858e-8,6.176061335580363e-8,
    6.023573837886479e-8,5.8748512699252157e-8,5.729800675250309e-8,
    5.588331392518268e-8,5.4503549988221496e-8,5.3157852544244216e-8,
    5.18453804885418e-8,5.05653134833552e-8,4.93168514451363e-8,
    4.809921404447013e-8,4.691164021834418e-8,4.5753387694457955e-8,
    4.4623732527280045e-8,4.352196864555741e-8,4.244740741099836e-8,
    4.139937718785167e-8,4.0377222923111285e-8,3.93803057370881e-8,
    3.840800252408829e-8,3.745970556295245e-8,3.6534822137210456e-8,
    3.5632774164613435e-8,3.475299783581462e-8,3.389494326196924e-8,
    3.305807413103648e-8,3.2241867372567335e-8,3.1445812830767746e-8,
    3.066941294563555e-8,2.991218244196846e-8,2.917364802605163e-8,
    2.8453348089834e-8,2.7750832422407467e-8,2.7065661928611266e-8,
    2.6397408354582483e-8,2.574565402008376e-8,2.510999155743982e-8,
    2.4490023656918752e-8,2.388536281840125e-8,2.329563110917976e-8,
    2.2720459927738556e-8,2.215948977336598e-8,2.1612370021454267e-8,
    2.1078758704348363e-8,2.0558322297604485e-8,2.0050735511526694e-8,
    1.9555681087850496e-8,1.9072849601445554e-8,1.860193926691551e-8,
    1.814265574997178e-8,1.7694711983465263e-8,1.7257827987960214e-8,
    1.6831730696737537e-8,1.6416153785119698e-8,1.6010837504008696e-8,
    1.5615528517534617e-8,1.522997974471263e-8,1.485395020500893e-8,
    1.4487204867720514e-8,1.4129514505072949e-8,1.3780655548945718e-8,
    1.3440409951135022e-8,1.3108565047066197e-8,1.2784913422871827e-8,
    1.2469252785750989e-8,1.2161385837529812e-8,1.186112015134383e-8,
    1.156826805136461e-8,1.1282646495496604e-8,1.1004076960969586e-8,
    1.0732385332756225e-8,1.0467401794744658e-8,1.0208960723597601e-8,
    9.956900585232694e-9,9.711063833858167e-9,9.471296813501711e-9,
    9.237449661970594e-9,9.009376217182673e-9,8.786933925810598e-9,
    8.569983754181119e-9,8.358390101374608e-9,8.152020714470167e-9,
    7.950746605883027e-9,7.754441972743351e-9,7.56298411826514e-9,
    7.376253375056834e-9,7.194133030325383e-9,7.016509252926757e-9,
    6.843271022217988e-9,6.674310058665473e-9,6.5095207561668245e-9,
    6.34880011604368e-9,6.192047682664017e-9,6.039165480654292e-9,
    5.890057953661489e-9,5.7446319046273555e-9,5.602796437537268e-9,
    5.464462900607116e-9,5.329544830873222e-9,5.197957900150024e-9,
    5.069619862322287e-9,4.944450501938644e-9,4.822371584074214e-9,
    4.70330680543136e-9,4.587181746647508e-9,4.473923825780761e-9,
    4.363462252943702e-9,4.2557279860574625e-9,4.1506536876982236e-9,
    4.04817368300902e-9,3.9482239186509444e-9,3.850741922767617e-9,
    3.755666765938289e-9,3.6629390230950104e-9,3.572500736379934e-9,
    3.484295378919872e-9,3.398267819495071e-9,3.3143642880804447e-9,
    3.2325323422375904e-9,3.152720834336478e-9,3.074879879586606e-9,
    2.9989608248573072e-9,2.92491621826799e-9,2.8526997795292008e-9,
    2.782266371015863e-9,2.7135719695548735e-9,2.646573638909117e-9,
    2.5812295029409497e-9,2.517498719438278e-9,2.455341454586782e-9,
    2.394718858072556e-9,2.335593038799337e-9,2.277927041205363e-9,
    2.221684822164966e-9,2.16683122846039e-9,2.11333197480995e-9,
    2.061153622438558e-9,2.0102635581774248e-9,1.9606299740797898e-9,
    1.912221847539868e-9,1.865008921902767e-9,1.8189616875530459e-9,
    1.774051363470265e-9,1.7302498792399302e-9,1.6875298575085259e-9,
    1.6458645968718275e-9,1.6052280551856116e-9,1.5655948332884871e-9,
    1.5269401591266087e-9,1.4892398722702979e-9,1.4524704088130316e-9,
    1.4166087866431961e-9,1.381632591079536e-9,1.3475199608612614e-9,
    1.3142495744840163e-9,1.2818006368732813e-9,1.2501528663867426e-9,
    1.2192864821376192e-9,1.1891821916309783e-9,1.1598211787052668e-9,
    1.1311850917716326e-9,1.103256032343554e-9,1.0760165438497166e-9,
    1.0494496007230992e-9,1.0235385977594125e-9,9.98267339738337e-10,
    9.736200313009565e-10,9.495812670771565e-10,9.261360220567754e-10,
    9.032696421984617e-10,8.809678352704464e-10,8.592166619174109e-10,
    8.38002526947946e-10,8.173121708371228e-10,7.971326614388382e-10,
    7.77451385902825e-10,7.582560427911907e-10,7.395346343896262e-10,
    7.212754592084525e-10,7.034671046687896e-10,6.860984399693441e-10,
    6.691586091292782e-10,6.526370242028764e-10,6.365233586617411e-10,
    6.208075409403602e-10,6.054797481410691e-10,5.905303998944039e-10,
    5.75950152371066e-10,5.6172989244173e-10,5.478607319810273e-10,
    5.343340023121944e-10,5.211412487888539e-10,5.082742255105915e-10,
    4.957248901690047e-10,4.834853990209858e-10,4.715481019861416e-10,
    4.5990553786523166e-10,4.485504296766812e-10,4.3747568010823535e-10,
    4.266743670808971e-10,4.161397394224149e-10,4.058652126475693e-10,
    3.9584436484266053e-10,3.86070932651607e-10,3.7653880736113435e-10,
    3.6724203108264176e-10,3.581747930283181e-10,3.493314258792144e-10,
    3.407064022429891e-10,3.3229433119909875e-10,3.240899549293071e-10,
    3.1608814543136926e-10,3.082839013138669e-10,3.0067234467017916e-10,
    2.9324871802962433e-10,2.860083813838946e-10,2.7894680928689246e-10,
    2.7205958802618263e-10,2.6534241286428096e-10,2.5879108534804686e-10,
    2.524015106845207e-10,2.4616969518153846e-10,2.400917437515472e-10,
    2.3416385747705087e-10,2.2838233123615701e-10,2.2274355138676214e-10,
    2.172439935079017e-10,2.1188022019687482e-10,2.0664887892075802e-10,
    2.0154669992095767e-10,1.9657049416951026e-10,1.917171513758312e-10,
    1.8698363804268407e-10,1.8236699557014806e-10,1.7786433840639205e-10,
    1.7347285224411606e-10,1.6918979226151304e-10,1.650124814066678e-10,
    1.609383087243143e-10,1.5696472772389927e-10,1.5308925478794762e-10,
    1.493094676197164e-10,1.4562300372918248e-10,1.4202755895641073e-10,
    1.385208860313755e-10,1.3510079316934772e-10,1.3176514270095466e-10,
    1.2851184973606853e-10,1.2533888086068347e-10,1.2224425286596262e-10,
    1.192260315086714e-10,1.162823303022097e-10,1.1341130933749743e-10,
    1.106111741329728e-10,1.0788017451298005e-10,1.052166035138559e-10,
    1.026187963170189e-10,1.0008512920840488e-10,9.761401856359382e-11,
    9.52039198579905e-11,9.285332670144929e-11,9.056076989672867e-11,
    8.832481652119633e-11,8.614406903120674e-11,8.401716438858868e-11,
    8.19427732087045e-11,7.991959892953932e-11,7.794637700130845e-11,
    7.602187409607351e-11,7.414488733687029e-11,7.23142435458737e-11,
    7.05287985111216e-11,6.878743627134586e-11,6.708906841846097e-11,
    6.543263341727182e-11,6.38170959419816e-11,6.224144622907783e-11,
    6.070469944619797e-11,5.92058950765778e-11,5.7744096318695667e-11,
    5.631838950074272e-11,5.4927883509546835e-11,5.357170923359851e-11,
    5.224901901982845e-11,5.095898614379546e-11,4.9700804292958246e-11,
    4.8473687062702556e-11,4.727686746481326e-11,4.610959744808222e-11,
    4.497114743075081e-11,4.386080584449881e-11,4.277787868968998e-11,
    4.172168910160013e-11,4.069157692735513e-11,3.968689831331278e-11,
    3.8707025302634525e-11,3.775134544279098e-11,3.6819261402759586e-11,
    3.5910190599673686e-11,3.502356483468833e-11,3.4158829937838525e-11,
    3.331544542166412e-11,3.249288414338788e-11,3.1690631975434486e-11,
    3.090818748408321e-11,3.014506161605645e-11,2.9400777392844726e-11,
    2.867486961257992e-11,2.796688455926927e-11,2.7276379719207388e-11,
    2.6602923504391594e-11,2.5946094982764667e-11,2.5305483615118915e-11,
    2.4680688998496078e-11,2.4071320615921796e-11,2.347699759232051e-11,
    2.289734845645553e-11,2.2332010908747674e-11,2.178063159482649e-11,
    2.1242865884671703e-11,2.0718377657208856e-11,2.020683909022217e-11,
    1.9707930455455216e-11,1.9221339918770554e-11,1.8746763345242746e-11,
    1.8283904109064703e-11,1.783247290814639e-11,1.7392187583291844e-11,
    1.6962772941840653e-11,1.6543960585663154e-11,1.6135488743403355e-11,
    1.57371021068629e-11,1.5348551671425312e-11,1.496959458042016e-11,
    1.459999397332936e-11,1.4239518837742066e-11,1.3887943864964021e-11,
    1.3545049309192453e-11,1.3210620850167937e-11,1.2884449459216921e-11,
    1.2566331268602371e-11,1.2256067444099483e-11,1.1953464060717953e-11,
    1.1658331981492679e-11,1.1370486739266698e-11,1.1089748421393543e-11,
    1.0815941557285693e-11,1.0548895008739873e-11,1.0288441862970217e-11,
    1.0034419328282077e-11,9.786668632322207e-12,9.545034922840628e-12,
    9.309367170903036e-12,9.079518076492902e-12,8.85534397644393e-12,
    8.636704754646167e-12,8.423463754468647e-12,8.215487693344648e-12,
    8.012646579465829e-12,7.814813630532921e-12,7.62186519451289e-12,
    7.433680672352188e-12,7.250142442598514e-12,7.0711357878836475e-12,
    6.89654882322118e-12,6.726272426074961e-12,6.560200168153779e-12,
    6.398228248890302e-12,6.2402554305624016e-12,6.086182975016128e-12,
    5.935914581951324e-12,5.789356328730652e-12,5.6464166116749505e-12,
    5.50700608880802e-12,5.3710376240148315e-12,5.238426232578777e-12,
    5.109089028063325e-12,4.98294517050535e-12,4.859915815887585e-12,
    4.739924066858413e-12,4.622894924668662e-12,4.508755242294825e-12,
    4.39743367871984e-12,4.288860654342681e-12,4.182968307488728e-12,
    4.079690451994141e-12,3.97896253583724e-12,3.8807216007914416e-12,
    3.7849062430743565e-12,3.69145657496833e-12,3.6003141873887774e-12,
    3.5114221133765118e-12,3.42472479249158e-12,3.340168036086218e-12,
    3.2576989934350907e-12,3.177266118701971e-12,3.0988191387218256e-12,
    3.0223090215784907e-12,2.947687945958158e-12,2.8749092712594217e-12,
    2.8039275084414685e-12,2.7346982915918745e-12,2.6671783501964972e-12,
    2.601325482094023e-12,2.537098527098176e-12,2.474457341271338e-12,
    2.413362771833214e-12,2.3537766326891e-12,2.2956616805623547e-12,
    2.2389815917160797e-12,2.183700939249669e-12,2.129785170955785e-12,
    2.0772005877241296e-12,2.0259143224784246e-12,1.9758943196333673e-12,
    1.9271093150589045e-12,1.8795288165390832e-12,1.8331230847134457e-12,
    1.787863114488979e-12,1.7437206169109434e-12,1.7006680014814045e-12,
    1.6586783589142312e-12,1.6177254443159347e-12,1.5777836607817719e-12,
    1.5388280433968073e-12,1.5008342436320735e-12,1.463778514125909e-12,
    1.4276376938411008e-12,1.3923891935884977e-12,1.3580109819079959e-12,
    1.3244815712981989e-12,1.2917800047859961e-12,1.2598858428277863e-12,
    1.2287791505341117e-12,1.1984404852096714e-12,1.1688508842010384e-12,
    1.1399918530443554e-12,1.1118453539057038e-12,1.084393794306884e-12,
    1.0576200161295145e-12,1.0315072848906821e-12,1.006039279283317e-12,
    9.812000809748568e-13,9.56974164657782e-13,9.333463883457665e-13,
    9.103019839094727e-13,8.878265478459658e-13,8.659060322760654e-13,
    8.445267361639731e-13,8.236752967536578e-13,8.033386812167219e-13,
    7.835041785064389e-13,7.641593914129444e-13,7.452922288145841e-13,
    7.268908981205505e-13,7.089438979001508e-13,6.914400106940203e-13,
    6.743682960028529e-13,6.5771808344924e-13,6.414789661083205e-13,
    6.256407940031327e-13,6.101936677605324e-13,5.951279324237693e-13,
    5.804341714178307e-13,5.661032006637615e-13,5.521260628383316e-13,
    5.384940217754036e-13,5.251985570055508e-13,5.122313584304917e-13,
    4.995843211289951e-13,4.872495402910552e-13,4.752193062771153e-13,
    4.634860997992977e-13,4.5204258722160863e-13,4.4088161597616605e-13,
    4.299962100926243e-13,4.1937956583795446e-13,4.0902504746389483e-13,
    3.989261830593964e-13,3.8907666050545765e-13,3.794703235298559e-13,
    3.701011678592672e-13,3.609633374664043e-13,3.5205112090981264e-13,
    3.4335894776402454e-13,3.3488138513787214e-13,3.2661313427874473e-13,
    3.1854902726069984e-13,3.1068402375434455e-13,3.0301320787645753e-13,
    2.955317851174111e-13,2.8823507934443873e-13,2.8111852987890343e-13,
    2.741776886457279e-13,2.674082173931959e-13,2.608058849814115e-13,
    2.543665647376923e-13,2.480862318772673e-13,2.419609609876585e-13,
    2.359869235751635e-13,2.301603856719299e-13,2.2447770550209742e-13,
    2.1893533120557214e-13,2.1352979861800073e-13,2.0825772910554945e-13,
    2.0311582745315406e-13,1.9810087980489796e-13,1.9320975165524992e-13,
    1.8843938588989806e-13,1.8378680087494912e-13,1.7924908859331557e-13,
    1.7482341282710566e-13,1.7050700738489696e-13,1.6629717437277835e-13,
    1.621912825080744e-13,1.5818676547471274e-13,1.5428112031918877e-13,
    1.504719058861403e-13,1.4675674129254726e-13,1.4313330443959874e-13,
    1.3959933056130979e-13,1.3615261080896538e-13,1.3279099087051956e-13,
    1.2951236962408174e-13,1.2631469782464381e-13,1.2319597682323895e-13,
    1.2015425731771786e-13,1.171876381343728e-13,1.1429426503964337e-13,
    1.1147232958115728e-13,1.0872006795739197e-13,1.0603575991523846e-13,
    1.0341772767478841e-13,1.008643348806681e-13,9.837398557926047e-14,
    9.594512322118514e-14,9.357622968840175e-14, };

#define myexp exp
#undef myexp
// use look-up table instead of exp() function
#define myexp(x) ((x)>0||x< -0.025*(sizeof(exptab)/sizeof(*exptab)+1)?\
    1e-20:exptab[(int)(-x*40.+0.5)])
// #define myexp(x) ((1+0.158206*x)/(1+((-0.679137+0.670515*x)*x)))
//  (((0.866887*x+0.210165)*x+.377786)*x+1))


#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************
functions for statex:  
  odpScore: compute sum of normal densities to be used as numerator 
  or denominator in score nullODPScore: odpScore on null permutations

***********************************************************************/

/*
  Compute numerators or denominators of ODP statistics

  gah - change to four separate routines for numerator and denominator
        for lr, and numerator and denominator for lr0.  A few extra
        arguments for some, if they are not all needed.

  xx: m-vector of sum of squared data for each gene
  xk: (m x k) matrix of group-specific sums of squares, collapsed by columns
  mu: (m x k) matrix of group-specific means for each gene, collapsed by columns
  sigma: m-vector of pooled sd's
  m: number of genes
  n: number of samples
  k: number of comparison groups
  nGrp: number of samples in each comparison group
  *null: compute numerator (*null = 0) or denominator (*null = 1)
  m0: number of genes to use in denominator
  nii: m0-vector of indices for genes to use in denominator
  scr: output
*/

void odp(double *xx, double *xk, double *mu, double *sigma, long *m, 
     long *n, long *k, int *nGrp, long *null, long *m0, int *nii, 
     double *scr, update_progress_routine* callback, void* callbackArg, 
     double *currentStart, double *maxStart) {
//   assert(0);
}

/* gah - size parameters first, and passed by value */
/* odp?t routines return the proper maxStart value for the
   corresponding routine.  This allows convenient calculation
   of maxStart in the general case 
 */

#define ENOUGH 1000000

double  odp1t(long mi, long mj, long n, long k) {
  return 1834+367.*mj+10.0*mi+445.*(mi)*(mj);
  }
   
void odp1(long mi, long mj, long n, long k,
     double *xx, double *xk, double *mu, double *sigma, 
     int *nGrp, int *nii, 
     double *scr, update_progress_routine* callback, void* callbackArg, 
     double *currentStart, double *maxStart) {
  long i, j, l;
  int iCurrent = 0;
#ifdef EDGE_DEBUG
/* gah - for rdtsc timing */
  long long t, t1, t2, t3, t4, t5, timing=0,rdtsc();
#endif

  /* tracking thread progress: */ 
  double max, current;

  /* temporaries for i and j loops */
  double middlet, lastt, powt, sigma2t, xt;

#ifdef EDGE_DEBUG
timing= - rdtsc();
#endif
//  max = maxStart ? *maxStart : (double)(mi) * (mj);
  max = *maxStart;
  current = *currentStart+1834+10.0*mi;

/*  gah - I think the following is wrong: increment is outside k loop */
/*  if(!maxStart && *null == 0) max = max * (k);   */

#ifndef EDGE_DEBUG 
  do_callback(callback, callbackArg, max, current, ODP_INIT);
#else
  t1= -timing;
  do_callback(callback, callbackArg, max, current, ODP_INIT);
  t2=rdtsc();
#endif

  /* compute numerator if alternative component */

/* exchange the i and j loops */
  for(i = 0; i < mi; i++) scr[i] = 1e-10;
#ifdef EDGE_DEBUG
  t4=t5=0;
  t3=rdtsc();
#endif
  for(j = 0; j < mj; j++) {
    current += 367.;
#ifdef EDGE_DEBUG
  t4-=rdtsc();
#endif
    powt = pow(1 / sigma[j], n);
    sigma2t = -0.5/(sigma[j]*sigma[j]);
/* lastt loop moved out of i loop */
    lastt = 0;
    for(l = 0; l < k; l++) {
      lastt += nGrp[l] * mu[j + l * mj] * mu[j + l * mj];
    }
#ifdef EDGE_DEBUG
  t=rdtsc();
  t4+=t;
  t5-=t;
#endif
    for(i = 0; i < mi; i++, iCurrent++) {
      current += 445;

      if (iCurrent > ENOUGH) {
        do_callback(callback, callbackArg, max, current, ODP_COMPUTING_ODP);
        iCurrent = 0;
      }

/* sum over l for each i and j, so zero middlet here */
/* lastt has no i dependence so move it out of the loop */
      middlet=0;
      for(l = 0; l < k; l++) {
        middlet += 2 * xk[i + l * mi] * mu[j + l * mj];
      }
      xt = sigma2t * (xx[i] - middlet + lastt);
#ifdef EDGE_DEBUG2
      fprintf(stderr,"exp1 %g %g %g\n",xt,myexp(xt),exp(xt));
#endif
      if(xt> -30.) scr[i] += powt * myexp(xt);
    }
#ifdef EDGE_DEBUG
  t5+=rdtsc();
#endif
  }
#ifdef EDGE_DEBUG
  timing += rdtsc();
#endif
#ifdef EDGE_DEBUG
  for(i=0;i<10 && i<mi;i++) fprintf(stderr,"num[%ld]=%g\n",i,scr[i]);
#endif

  *currentStart = current;

#ifdef EDGE_DEBUG
fprintf(stderr,"odp1 timing=%lld current=%.0f\n",timing,current);
fprintf(stderr,"odpx t2-t1=%lld t3-t2=%lld t4=%lld t5=%lld\n",
         t2-t1,t3-t2,t4,t5);
#endif
}

double  odp2t(long mi, long mj, long n, long k) {
  return 1834+367.*mj+10.0*mi+386.0*(mi)*(mj);
  }
   
void odp2(long mi, long mj, long n, long k,
     double *xx, double *xk, double *mu, double *sigma, 
     int *nGrp, int *nii, 
     double *scr, update_progress_routine* callback, void* callbackArg, 
     double *currentStart, double *maxStart) {
  long i, j;
  int iCurrent = 0;
#ifdef EDGE_DEBUG
/* gah - for rdtsc timing */
  long long timing=0,rdtsc();
#endif

  /* tracking thread progress: */ 
  double max, current;

  /* temporaries for i and j loops */
  double powt, sigma2t, xt;

#ifdef EDGE_DEBUG
timing -= rdtsc();
#endif
//  max = maxStart ? *maxStart : (double)(mi) * (mj);
  max = *maxStart;
  current = *currentStart+1834+10.0*mi;

/*  gah - I think the following is wrong: increment is outside k loop */
/*  if(!maxStart && *null == 0) max = max * (k);   */

  do_callback(callback, callbackArg, max, current, ODP_INIT);

  /* compute denominator if null component */
  for(i = 0; i < mi; i++) scr[i] = 1e-10;
  for(j = 0; j < mj; j++) {
    current += 367.;
    powt = pow(sigma[nii[j]],-n); 
    sigma2t= -0.5 / (sigma[nii[j]] * sigma[nii[j]]);
    for(i = 0; i < mi; i++, iCurrent++) {
      current += 386.0;

      if (iCurrent > ENOUGH)
        {
        do_callback(callback, callbackArg, max, current, ODP_COMPUTING_ODP);
        iCurrent = 0;
      }

      xt = sigma2t * xx[i];
      if(xt> -30.) scr[i] += powt * myexp(xt);
    }
  }
#ifdef EDGE_DEBUG
  for(i=0;i<10 && i<mi;i++) fprintf(stderr,"den[%ld]=%g\n",i,scr[i]);
#endif

  *currentStart = current;
#ifdef EDGE_DEBUG
timing += rdtsc();
#endif
#ifdef EDGE_DEBUG
fprintf(stderr,"odp2 timing=%lld current=%.0f\n",timing,current);
#endif
}

double  odp3t(long mi, long mj, long n, long k) {
  return 1834+367.*mj+10.0*mi+445.*(mi)*(mj);
  }
   
void odp3(long mi, long mj, long n, long k,
     double *xx, double *xk, double *mu, double *sigma, 
     int *nGrp, int *nii, 
     double *scr, update_progress_routine* callback, void* callbackArg, 
     double *currentStart, double *maxStart) {
  long i, j, l;
  int iCurrent = 0;
#ifdef EDGE_DEBUG
/* gah - for rdtsc timing */
  long long timing=0,rdtsc();
#endif

  /* tracking thread progress: */ 
  double max, current;

  /* temporaries for i and j loops */
  double middlet, lastt, powt, sigma2t, xt;

#ifdef EDGE_DEBUG
timing -= rdtsc();
#endif
//  max = maxStart ? *maxStart : (double)(mi) * (mj);
  max = *maxStart;
  current = *currentStart+1834+10.0*mi;

/*  gah - I think the following is wrong: increment is outside k loop */
/*  if(!maxStart && *null == 0) max = max * (k);   */

  do_callback(callback, callbackArg, max, current, ODP_INIT);

  /* compute numerator if alternative component */

/* exchange the i and j loops */
  for(i = 0; i < mi; i++) scr[i] = 1e-10;
  for(j = 0; j < mj; j++) {
    current += 367.;
    powt = pow(1 / sigma[j], n);
    sigma2t = -0.5/(sigma[j]*sigma[j]);
/* lastt loop moved out of i loop */
    lastt = 0;
    for(l = 0; l < k; l++) {
      lastt += nGrp[l] * mu[j + l * mj] * mu[j + l * mj];
    }

    for(i = 0; i < mi; i++, iCurrent++) {

      current += 445;
      if (iCurrent > ENOUGH) {
        do_callback(callback, callbackArg, max, current, ODP_COMPUTING_ODP);
        iCurrent = 0;
      }

/* sum over l for each i and j, so zero middlet here */
/* lastt has no i dependence so move it out of the loop */
      middlet=0;
      for(l = 0; l < k; l++) {
        middlet += 2 * xk[i + l * mi] * mu[j + l * mj];
      }

      xt = sigma2t * (xx[i] - middlet + lastt);
      if(xt> -30.) scr[i] += powt * myexp(xt);
    }
  }
#ifdef EDGE_DEBUG
  for(i=0;i<10 && i<mi;i++) fprintf(stderr,"num0[%ld]=%g\n",i,scr[i]);
#endif

  *currentStart = current;
#ifdef EDGE_DEBUG
timing += rdtsc();
#endif
#ifdef EDGE_DEBUG
fprintf(stderr,"odp3 timing=%lld current=%.0f\n",timing,current);
#endif
}

double  odp4t(long mi, long mj, long n, long k) {
  return 1834+367.*mj+10.0*mi+373.0*(mi)*(mj);
  }
   
void odp4(long mi, long mj, long n, long k,
     double *xx, double *xk, double *mu, double *sigma, 
     int *nGrp, int *nii, 
     double *scr, update_progress_routine* callback, void* callbackArg, 
     double *currentStart, double *maxStart) {
  long i, j;
  int iCurrent = 0;
#ifdef EDGE_DEBUG
/* gah - for rdtsc timing */
  long long timing=0,rdtsc();
#endif

  /* tracking thread progress: */ 
  double max, current;

  /* temporaries for i and j loops */
  double powt, sigma2t, xt;

#ifdef EDGE_DEBUG
timing -= rdtsc();
#endif
//  max = maxStart ? *maxStart : (double)(mi) * (mj);
  max = *maxStart;
  current = *currentStart+1834+10.0*mi;

/*  gah - I think the following is wrong: increment is outside k loop */
/*  if(!maxStart && *null == 0) max = max * (k);   */

  do_callback(callback, callbackArg, max, current, ODP_INIT);

  /* compute denominator if null component */
  for(i = 0; i < mi; i++) scr[i] = 1e-10;
  for(j = 0; j < mj; j++) {
    current += 367.;
#if 0
    powt = pow(sigma[nii[j]],-n); 
    sigma2t= -0.5 / (sigma[nii[j]] * sigma[nii[j]]);
#else
    powt = pow(sigma[j],-n); 
    sigma2t= -0.5 / (sigma[j] * sigma[j]);
#endif
    for(i = 0; i < mi; i++, iCurrent++) {
      current += 373.0;

      if (iCurrent > ENOUGH)
        {
        do_callback(callback, callbackArg, max, current, ODP_COMPUTING_ODP);
        iCurrent = 0;
      }

      xt = sigma2t * xx[i];
#ifdef EDGE_DEBUG2
      fprintf(stderr,"exp4 %g %g %g\n",xt,myexp(xt),exp(xt));
#endif
      if(xt> -30.) scr[i] += powt * myexp(xt);
    }
  }
#ifdef EDGE_DEBUG
  for(i=0;i<10 && i<mi;i++) fprintf(stderr,"den0[%ld]=%g\n",i,scr[i]);
#endif

  *currentStart = current;
#ifdef EDGE_DEBUG
timing += rdtsc();
#endif
#ifdef EDGE_DEBUG
fprintf(stderr,"odp4 timing=%lld current=%.0f\n",timing,current);
#endif
}

/*
  Compute ODP statistics under null hypothesis by bootstrap

  res: (m x n) matrix of residuals, collapsed by columns
  mu: (m x k) matrix of group-specific means, collapsed by columns
  sigma: m-vector of pooled sd's
  sigma0: m-vector of sd's computed under null hypothesis
         (but only m0 of them will be used)
  vv: (B x n) matrix of indices defining bootstrap samples, collapsed by columns
  ID: (n x k) model matrix, collapsed by columns
  m0: number of genes (the subset of genes used in 
        denominators of original ODP statistics)
  n: number of samples
  k: number of comparison groups
  nGrp: number of samples in each comparison group
  B: number of bootstrap samples
  scr: (m x B) matrix of null ODP statistics, collapsed by columns
*/

void nullODP(long m, long m0, long n, long k, long B, double *res, 
     double *mu, double *sigma, double *sigma0, int *vv, int *ID, int *nii, 
      int *nGrp, double *scr, update_progress_routine* callback, 
      void* callbackArg, double *currentStart, double *maxStart) {

//   int ii;
  long i, j, l, h;
  double *num, *den, *dat0, *xx, *xk, *mu0;
  /* max and current for tracking thread progress: */ 
  double max = maxStart ? *maxStart : (double)(m0) * (m0+m) * (B);
  double current = currentStart ? *currentStart : 0;

  /* allocate memory */
//  pass nii instead of using all
//  nii = intvector(0, m0 - 1);
  num = vector(0, m - 1);
  den = vector(0, m - 1);
  dat0 = vector(0, m * n - 1);
  xx = vector(0, m - 1);
  xk = vector(0, m * k - 1);
  mu0 = vector(0, m0 - 1);

//  for(ii = 0; ii < m0; ii++)
//    nii[ii] = ii;

  /* loop over bootstrap samples */
  for(i = 0; i < B; i++) {
    /* initialization */
    for(j = 0; j < m0; j++) {
      xx[j] = 0.0;
      mu0[j] = 0.0;
      for(l = 0; l < k; l++)
        xk[j + l * m0] = 0.0;
    }

    /* form bootstrap sample */
    for(j = 0; j < m0; j++) {
      for(l = 0; l < n; l++) {
        dat0[j + l * m0] = res[j + vv[i + l * B] * m0];
        mu0[j] += dat0[j + l * m0] / n;
      }
      for(l = 0; l < n; l++) 
        dat0[j + l * m0] -= mu0[j];
    }

    /* initial computations on bootstrap sample */
    for(j = 0; j < m0; j++) {
      for(l = 0; l < n; l++)
        xx[j] += dat0[j + l * m0] * dat0[j + l * m0];
      for(l = 0; l < k; l++) {
        for(h = 0; h < n; h++)
          xk[j + l * m0] += dat0[j + h * m0] * ID[h + l * n];
      }
    }

    /* compute scores */
    odp3(m0, m , n, k, xx, xk, mu, sigma, nGrp, nii, num, 
          callback, callbackArg, &current, &max);
    odp4(m0, m0, n, k, xx, xk, mu, sigma0, nGrp, nii, den, 
          callback, callbackArg, &current, &max);

    /* prepare output */
    for(j = 0; j < m0; j++)
      scr[j + i * m0] = num[j] / den[j];
  }

  /* free memory */
//  free_intvector(nii, 0, m0 - 1);
  free_vector(num, 0, - 1);
  free_vector(den, 0,  - 1);
  free_vector(dat0, 0,  - 1);
  free_vector(xx, 0, - 1);
  free_vector(xk, 0,  - 1);
  free_vector(mu0, 0,  - 1);

  if(currentStart) *currentStart = current;
}

/****************************************************************************
functions for timex:  
  nullFIndep: simulate null F statistics under independent sampling
  nullFLongNoInt: simulate null F statistics under longitudinal sampling, no intercept
  nullFLongInt: simulate null F statistics under longitudinal sampling, with intercept
  FNoEM: compute F statistics without using EM algorithm
  FEM: compute F statistics using EM algorithm
  EMFit: the EM algorithm.
  EMFitR: R wrapper for calling EMFit directly
*****************************************************************************/

/* simulate null F statistics under independent sampling */
void nullFIndep(double *fitR, double *resR, int *vR, double *H1R, 
        double *H0R, long *m, long *n, long *B, double *F, 
        update_progress_routine* callback, void* callbackArg) {

  long i, j, k, **v;
  double **fit, **res, **H0, **H1, **dat, **fit1, **fit0, *rss1, *rss0, *Ft;
  double max = (double)(*B) * (*m) * (*n + 1) * (*n + 1);
  double current = 0;
  int iCurrent = 0;

  /* allocate memory */
  v = imatrix(0, *B - 1, 0, *n - 1);
  fit = matrix(0, *m - 1, 0, *n - 1);
  res = matrix(0, *m - 1, 0, *n - 1);
  H0 = matrix(0, *n - 1, 0, *n - 1);
  H1 = matrix(0, *n - 1, 0, *n - 1);
  dat = matrix(0, *m - 1, 0, *n - 1);
  fit1 = matrix(0, *m - 1, 0, *n - 1);
  fit0 = matrix(0, *m - 1, 0, *n - 1);
  rss1 = vector(0, *m - 1);
  rss0 = vector(0, *m - 1);
  Ft = vector(0, *m - 1);

  /* reform matrix inputs (n*(B+m+n) calculations, not counted)*/
  for(i = 0; i < *n; i++) {
    for(j = 0; j < *B; j++)
      v[j][i] = vR[j + i * *B];

    for(j = 0; j < *m; j++) {
      fit[j][i] = fitR[j + i * *m];
      res[j][i] = resR[j + i * *m];
    }

    for(j = 0; j < *n; j++) {
      H0[j][i] = H0R[j + i * *n];
      H1[j][i] = H1R[j + i * *n];
    }
  }

  /* simulate null statistics (B*m*(n^2+2n+1) calculations)*/
  for(i = 0; i < *B; i++) {
    /* initializations (m*n, not counted) */
    for(j = 0; j < *m; j++) {
      for(k = 0; k < *n; k++) {
        fit1[j][k] = 0.0;
        fit0[j][k] = 0.0;
      }

      rss1[j] = 0.0;
      rss0[j] = 0.0;
    }

    /* form a null sample (m*n) */
    for(j = 0; j < *m; j++, current++, iCurrent++) 
      for(k = 0; k < *n; k++) 
    {
      if(iCurrent % 1000 == 0) 
        {
          do_callback(callback, callbackArg, max, current, TIMECOURSE);
          iCurrent = 0;
        }
      dat[j][k] = fit[j][k] + res[j][v[i][k]];
    }

    /* compute F statistics (m*(n^2+n+1)) */
    FNoEM(dat, H1, H0, fit1, fit0, rss1, rss0, m, n, Ft, callback, 
        callbackArg, &current, &max);
    for(j = 0; j < *m; j++)
      F[j + i * *m] = Ft[j]; 
  }

  /* free memory */
  free_imatrix(v, 0, *B - 1, 0, *n - 1);
  free_matrix(fit, 0, *m - 1, 0, *n - 1);
  free_matrix(res, 0, *m - 1, 0, *n - 1);
  free_matrix(H0, 0, *n - 1, 0, *n - 1);
  free_matrix(H1, 0, *n - 1, 0, *n - 1);
  free_matrix(dat, 0, *m - 1, 0, *n - 1);
  free_matrix(fit1, 0, *m - 1, 0, *n - 1);
  free_matrix(fit0, 0, *m - 1, 0, *n - 1);
  free_vector(rss1, 0,  - 1);
  free_vector(rss0, 0,  - 1);
  free_vector(Ft, 0,  - 1);
}

/* simulate null F statistics under longitudinal sampling, no intercept */
void nullFLongNoInt(double *fitR, double *resR, int *vR, int *rmv, 
       int *inc, double *H1R, double *H0R, double *wsqR, int *ind, 
       int *idx, long *m, long *n, long *N, int *nGrp, long *B, double *F,
       update_progress_routine* callback, void* callbackArg) {

  long i, j, k, l, ll, **v;
  double **fit, **res, **res0, **H0, **H1, **wsq, **dat, **fit1, 
       **fit0, *rss1, *rss0, *Ft, grpSum;
  double max = ((double)(*B)) * (*m) * ((double)(*n-*N)*(*n-*N) + *N + 
        (double)(*n + 1) * (*n + 1));
  double current = 0;
  int iCurrent = 0;

  /* allocate memory */
  v = imatrix(0, *B - 1, 0, *n - *N - 1);
  fit = matrix(0, *m - 1, 0, *n - 1);
  res = matrix(0, *m - 1, 0, *n - 1);
  res0 = matrix(0, *m - 1, 0, *n - *N - 1);
  H0 = matrix(0, *n - 1, 0, *n - 1);
  H1 = matrix(0, *n - 1, 0, *n - 1);
  wsq = matrix(0, *n - *N - 1, 0, *n - *N - 1);
  dat = matrix(0, *m - 1, 0, *n - 1);
  fit1 = matrix(0, *m - 1, 0, *n - 1);
  fit0 = matrix(0, *m - 1, 0, *n - 1);
  rss1 = vector(0, *m - 1);
  rss0 = vector(0, *m - 1);
  Ft = vector(0, *m - 1);

  /* reform matrix inputs */
  for(i = 0; i < *n - *N; i++) {
    for(j = 0; j < *B; j++)
      v[j][i] = vR[j + i * *B];
    for(j = 0; j < *m; j++)
      res0[j][i] = resR[j + i * *m];
    for(j = 0; j < *n - *N; j++)
      wsq[j][i] = wsqR[j + i * (*n - *N)];
  }

  for(i = 0; i < *n; i++) {
    for(j = 0; j < *m; j++)
      fit[j][i] = fitR[j + i * *m];
    for(j = 0; j < *n; j++) {
      H0[j][i] = H0R[j + i * *n];
      H1[j][i] = H1R[j + i * *n];
    }
  }

  /* simulate null statistics */
  for(i = 0; i < *B; i++) {
    /* initializations */
    for(j = 0; j < *m; j++) {
      for(k = 0; k < *n; k++) {
        fit1[j][k] = 0.0;
        fit0[j][k] = 0.0;
        res[j][k] = 0.0;
      }

      rss1[j] = 0.0;
      rss0[j] = 0.0;
    }

    /* do something mysterious to the residuals (m*((n-N)*(n-N) + N) */
    for(j = 0; j < *m; j++) {
      for(k = 0; k < *n - *N; k++)
        for(l = 0; l < *n - *N; l++, current++, iCurrent++)
      {
        if(iCurrent % 1000 == 0) 
          {
        do_callback(callback, callbackArg, max, current, TIMECOURSE);
        iCurrent = 0;
          }
        res[j][inc[k]] += res0[j][v[i][l]] * wsq[l][k];
      }

      ll = 0;
      for(k = 0; k < *N; k++, current++, iCurrent++) {
    if(iCurrent % 1000 == 0) 
      {
        do_callback(callback, callbackArg, max, current, TIMECOURSE);
        iCurrent = 0;
      }
        grpSum = 0.0;
        for(l = 0; l < nGrp[k]; l++, ll++)
          grpSum += res[j][idx[ll]];
        res[j][rmv[k]] = -grpSum;
      } 
    }

    /* form a null sample (m*n) */
    for(j = 0; j < *m; j++) 
      for(k = 0; k < *n; k++)
    {
      if(iCurrent % 1000 == 0) 
        {
          do_callback(callback, callbackArg, max, current, TIMECOURSE);
          iCurrent = 0;
        }
      dat[j][k] = fit[j][k] + res[j][k];
    }

    /* compute F statistics m*(n^2 + n + 1) */
    FNoEM(dat, H1, H0, fit1, fit0, rss1, rss0, m, n, Ft, callback, callbackArg, &current, &max);
    for(j = 0; j < *m; j++)
      F[j + i * *m] = Ft[j];
  }


  /* free_memory */
  free_imatrix(v, 0, *B - 1, 0, *n - *N - 1);
  free_matrix(fit, 0, *m - 1, 0, *n - 1);
  free_matrix(res, 0, *m - 1, 0, *n - 1);
  free_matrix(res0, 0, *m - 1, 0, *n - *N - 1);
  free_matrix(H0, 0, *n - 1, 0, *n - 1);
  free_matrix(H1, 0, *n - 1, 0, *n - 1);
  free_matrix(dat, 0, *m - 1, 0, *n - 1);
  free_matrix(wsq, 0, *n - *N - 1, 0, *n - *N - 1);
  free_matrix(fit1, 0, *m - 1, 0, *n - 1);
  free_matrix(fit0, 0, *m - 1, 0, *n - 1);
  free_vector(rss1, 0,  - 1);
  free_vector(rss0, 0,  - 1);
  free_vector(Ft, 0,  - 1);
}

/* simulate null F statistics under longitudinal sampling, with intercept */
void nullFLongInt(double *fitR, double *resR, int *vR, int *v1R, double *H1R,
          double *H0R, double *alphaR, int *ind, int *idx, 
          double *eps, long *m, long *n, long *N, int *nGrp, 
          long *B, double *F,
          update_progress_routine* callback, void* callbackArg) {

  long i, j, k, **v, **v1, *EM;
  double **fit, **res, **H0, **H1, **alpha, **alpha1, **alpha0, **dat, 
          **fit1, **fit0, *rss1, *rss0, **datC, *gm, **A0, **AA0, **AA, 
          *lik0, *lik, *tau, *sig, *atmp, *Ft;

  double max = ((double)*B) * 2;
  double current = 0;

  /* allocate memory */
  v = imatrix(0, *B - 1, 0, *n - 1);
  v1 = imatrix(0, *B - 1, 0, *n - 1);
  fit = matrix(0, *m - 1, 0, *n - 1);
  res = matrix(0, *m - 1, 0, *n - 1);
  H0 = matrix(0, *n - 1, 0, *n - 1);
  H1 = matrix(0, *n - 1, 0, *n - 1);
  alpha = matrix(0, *m - 1, 0, *n - 1);
  alpha1 = matrix(0, *m - 1, 0, *n - 1);
  alpha0 = matrix(0, *m - 1, 0, *n - 1);
  dat = matrix(0, *m - 1, 0, *n - 1);
  fit1 = matrix(0, *m - 1, 0, *n - 1);
  fit0 = matrix(0, *m - 1, 0, *n - 1);
  rss1 = vector(0, *m - 1);
  rss0 = vector(0, *m - 1);
  datC = matrix(0, *m - 1, 0, *n - 1);
  gm = vector(0, *m - 1);
  A0 = matrix(0, *m - 1, 0, *N - 1);
  AA0 = matrix(0, *m - 1, 0, *N - 1);
  AA = matrix(0, *m - 1, 0, *N - 1);
  lik0 = vector(0, *m - 1);
  lik = vector(0, *m - 1);
  tau = vector(0, *m - 1);
  sig = vector(0, *m - 1);
  EM = ivector(0, *m - 1);
  atmp = vector(0, *N - 1);
  Ft = vector(0, *m - 1);

  /* debug */
  saveDblArray("fitR.txt", fitR, (*m) * (*n));
  saveDblArray("resR.txt", resR, (*m) * (*n));
  saveDblArray("H1R.txt", H1R, (*n) * (*n));
  saveDblArray("H0R.txt", H0R, (*n) * (*n));
  saveDblArray("alphaR.txt", alphaR, (*m) * (*n));
  saveIntArray("vR.txt", vR, (*B) * (*n));
  saveIntArray("v1R.txt", v1R, (*B) * (*n));
  saveIntArray("ind.txt", ind, (*n));
  saveIntArray("idx.txt", idx, (*n));

  /* reform matrix inputs */
  for(i = 0; i < *n; i++) {
    for(j = 0; j < *B; j++) {
      v[j][i] = vR[j + i * *B];
      v1[j][i] = v1R[j + i * *B];
    }

    for(j = 0; j < *m; j++) {
      fit[j][i] = fitR[j + i * *m];
      res[j][i] = resR[j + i * *m];
      alpha[j][i] = alphaR[j + i * *m];
    }

    for(j = 0; j < *n; j++) {
      H0[j][i] = H0R[j + i * *n];
      H1[j][i] = H1R[j + i * *n];
    }
  }

  /* simulate null statistics */
  for(i = 0; i < *B; i++) {
    /* initializations */
    for(j = 0; j < *m; j++) {
      for(k = 0; k < *n; k++) {
        fit1[j][k] = 0.0;
        fit0[j][k] = 0.0;
      }

      rss1[j] = 0.0;
      rss0[j] = 0.0;
    }

    /* form a null sample */
    for(j = 0; j < *m; j++) 
      for(k = 0; k < *n; k++)
        dat[j][k] = fit[j][k] + res[j][v[i][k]] + alpha[j][v1[i][k]];

    /* compute F statistics */
    FEM(dat, H1, H0, ind, idx, eps, fit1, fit0, alpha1, alpha0, rss1, 
          rss0, datC, gm, A0, AA0, AA, lik0, lik, tau, sig, EM, atmp, 
          m, n, N, nGrp, Ft, callback, callbackArg, &max, &current);

    for(j = 0; j < *m; j++)
      F[j + i * *m] = Ft[j];
  }

  free_imatrix(v, 0, *B - 1, 0, *n - 1);
  free_imatrix(v1, 0, *B - 1, 0, *n - 1);
  free_matrix(fit, 0, *m - 1, 0, *n - 1);
  free_matrix(res, 0, *m - 1, 0, *n - 1);
  free_matrix(alpha, 0, *m - 1, 0, *n - 1);
  free_matrix(alpha1, 0, *m - 1, 0, *n - 1);
  free_matrix(alpha0, 0, *m - 1, 0, *n - 1);
  free_matrix(H0, 0, *n - 1, 0, *n - 1);
  free_matrix(H1, 0, *n - 1, 0, *n - 1);
  free_matrix(dat, 0, *m - 1, 0, *n - 1);
  free_matrix(fit1, 0, *m - 1, 0, *n - 1);
  free_matrix(fit0, 0, *m - 1, 0, *n - 1);
  free_vector(rss1, 0,  - 1);
  free_vector(rss0, 0,  - 1);
  free_matrix(datC, 0, *m - 1, 0, *n - 1);
  free_vector(gm, 0,  - 1);
  free_matrix(A0, 0, *m - 1, 0, *N - 1);
  free_matrix(AA0, 0, *m - 1, 0, *N - 1);
  free_matrix(AA, 0, *m - 1, 0, *N - 1);
  free_vector(lik0, 0,  - 1);
  free_vector(lik, 0,  - 1);
  free_vector(tau, 0,  - 1);
  free_vector(sig, 0,  - 1);
  free_ivector(EM, 0, *m - 1);
  free_vector(atmp, 0,  - 1);
  free_vector(Ft, 0,  - 1);

  saveDblArray("F.txt", F, (*m) * (*B));

}

/* compute F statistics without using EM algorithm */
void FNoEM(double **dat, double **H1, double **H0, double **fit1, 
       double **fit0, double *rss1, double *rss0, long *m, long *n, 
       double *F, update_progress_routine* callback, 
       void* callbackArg, double *pCurrent, double *pMax) {

  long j, k, l;
  double current = pCurrent == NULL ? 0 : *pCurrent;
  double max = pMax == NULL ? ((double)(*m))*((*n)*(*n) + (*n) + 1) : *pMax;
  int iCurrent = 0;

  /* compute fitted values under two models (m * n * n calculations) */
  for(j = 0; j < *m; j++) {
    for(k = 0; k < *n; k++) {
      for(l = 0; l < *n; l++, current++, iCurrent++) {
    if(iCurrent % 1000 == 0) 
      {
        do_callback(callback, callbackArg, max, current, TIMECOURSE);
        iCurrent = 0;
      }
        fit1[j][k] += dat[j][l] * H1[k][l];
        fit0[j][k] += dat[j][l] * H0[k][l];
      }
    }
  }

  /* compute residual sums-of-squares (m * n calculations) */
  for(j = 0; j < *m; j++) {
    for(k = 0; k < *n; k++, current++, iCurrent++) {
      if(iCurrent % 1000 == 0) 
    {
      do_callback(callback, callbackArg, max, current, TIMECOURSE);
      iCurrent = 0;
    }
      rss1[j] += (dat[j][k] - fit1[j][k]) * (dat[j][k] - fit1[j][k]);
      rss0[j] += (dat[j][k] - fit0[j][k]) * (dat[j][k] - fit0[j][k]);
    }
  }

  /* compute F statistics (m calculations) */
  for(j = 0; j < *m; j++)
    F[j] = rss0[j] / rss1[j] - 1.0;

  if(NULL != pCurrent)
    *pCurrent = current;
}

/* compute F statistics using EM algorithm */
void FEM(double **dat, double **H1, double **H0, int *ind, int *idx, 
     double *eps, double **fit1, double **fit0, double **alpha1, 
     double **alpha0, double *rss1, double *rss0, double **datC, 
     double *gm, double **A0, double **AA0, double **AA, 
     double *lik0, double *lik, double *tau, double *sig, 
     long *EM, double *atmp, long *m, long *n, long *N, int *nGrp, 
     double *F, update_progress_routine* callback, void* callbackArg, 
     double *pMax, double *pCurrent) {

  long j, k, iter;

  double max = pMax == NULL ? 2 : *pMax;
  double current = pCurrent == NULL ? 0 : *pCurrent;

  /* compute fitted values under two models */
  EMFit(dat, H1, ind, idx, eps, fit1, alpha1, datC, gm, A0, AA0, AA, 
    lik0, lik, tau, sig, EM, atmp, m, n, N, nGrp, &iter,
    callback, callbackArg, &max, &current);
  current++;
  if(NULL != callback) do_callback(callback, callbackArg, max, current, 
        TIMECOURSE);

  EMFit(dat, H0, ind, idx, eps, fit0, alpha0, datC, gm, A0, AA0, 
        AA, lik0, lik, tau, sig, EM, atmp, m, n, N, nGrp, &iter,
        callback, callbackArg, &max, &current);
  current++;
  if(NULL != callback) do_callback(callback, callbackArg, max, current, 
         TIMECOURSE);

  /* compute residual sums-of-squares */
  for(j = 0; j < *m; j++) {
    for(k = 0; k < *n; k++) {
      rss1[j] += (dat[j][k] - fit1[j][k] - alpha1[j][k]) * 
          (dat[j][k] - fit1[j][k] - alpha1[j][k]);
      rss0[j] += (dat[j][k] - fit0[j][k] - alpha0[j][k]) * 
          (dat[j][k] - fit0[j][k] - alpha0[j][k]);
    }
  }

  /* compute F statistics */
  for(j = 0; j < *m; j++)
    F[j] = rss0[j] / rss1[j] - 1.0;

  if(NULL != pCurrent)
    *pCurrent = current;
}

/* the EM algorithm */
/* note: ind argument not used */
void EMFit(double **dat, double **H, int *ind, int *idx, double *eps, 
       double **fit, double **alpha, double **datC, double *gm, 
       double **A0, double **AA0, double **AA, double *lik0, double *lik, 
       double *tau, double *sig, long *EM, double *atmp, long *m, long *n, 
       long *N, int *nGrp, long *iter, update_progress_routine* callback, 
       void* callbackArg, double *pMax, double *pCurrent) {
  
  long i, j, k, l, ll, nEM = *m;

  /* initialization */
  (*iter) = 1;
  for(i = 0; i < *m; i++) {
    gm[i] = 0.0;
    lik0[i] = 0.0;
    lik[i] = 1.0;
    EM[i] = i;
    for(j = 0; j < *N; j++)
      A0[i][j] = 0.0;
  }

  for(i = 0; i < *m; i++) {
    for(j = 0; j < *n; j++)
      gm[i] += dat[i][j];
    gm[i] /= *n;
  }
  
  for(i = 0; i < *m; i++)
    for(j = 0; j < *n; j++)
      datC[i][j] = dat[i][j] - gm[i];

  for(i = 0; i < *m; i++) {
    ll = 0;
    for(j = 0; j < *N; j++) {
      for(k = 0; k < nGrp[j]; k++, ll++)
        A0[i][j] += datC[i][idx[ll]];
      A0[i][j] /= nGrp[j];
      AA0[i][j] = A0[i][j] * A0[i][j];
      alpha[i][j] = A0[i][j];
      AA[i][j] = AA0[i][j];
    }
  }

  /* begin EM algorithm */
  for(; nEM > 1; (*iter)++) {
    for(i = 0; i < nEM; i++) {
      /* initializations */
      for(j = 0; j < *N; j++) {
        alpha[EM[i]][j] = A0[EM[i]][j];
        AA[EM[i]][j] = AA0[EM[i]][j];
        A0[EM[i]][j] = 0.0;
        AA0[EM[i]][j] = 0.0;
      } for(j = 0; j < *n; j++) {
        fit[EM[i]][j] = 0.0;
      }
      tau[EM[i]] = 0.0;
      sig[EM[i]] = 0.0;
    }

    for(i = 0; i < nEM; i++) {
      for(j = 0; j < *n; j++) {
        ll = 0;
        for(k = 0; k < *N; k++) 
          for(l = 0; l < nGrp[k]; l++, ll++) 
            fit[EM[i]][j] += H[j][ll] * (dat[EM[i]][ll] - alpha[EM[i]][k]);
      }

      for(j = 0; j < *N; j++)
        tau[EM[i]] += AA[EM[i]][j];
      tau[EM[i]] /= *N;

      ll = 0;
      for(j = 0; j < *N; j++)
        for(k = 0; k < nGrp[j]; k++, ll++)
          sig[EM[i]] += (dat[EM[i]][ll] - fit[EM[i]][ll]) * 
          (dat[EM[i]][ll] - fit[EM[i]][ll]) - 
            2 * (dat[EM[i]][ll] - fit[EM[i]][ll]) * alpha[EM[i]][j] + 
            AA[EM[i]][j];
      sig[EM[i]] /= *n;

      ll = 0;
      for(j = 0; j < *N; j++) {
        for(k = 0; k < nGrp[j]; k++, ll++)
          A0[EM[i]][j] += dat[EM[i]][idx[ll]] - fit[EM[i]][idx[ll]];
        A0[EM[i]][j] /= nGrp[j];
        A0[EM[i]][j] *= tau[EM[i]] * nGrp[j] / 
            (sig[EM[i]] + tau[EM[i]] * nGrp[j]);
      }

      for(j = 0; j < *N; j++) 
        AA0[EM[i]][j] = tau[EM[i]] * (1.0 - (tau[EM[i]] * nGrp[j] / 
         (sig[EM[i]] + tau[EM[i]] * nGrp[j]))) + A0[EM[i]][j] * A0[EM[i]][j];
    }

    for(i = 0; i < *m; i++)
      lik0[i] = lik[i];

    for(i = 0; i < nEM; i++)
      lik[EM[i]] = -*N / 2 * log(tau[EM[i]]) - *m / 2 * log(sig[EM[i]]);

    nEM = 0;
    for(i = 0; i < *m; i++) 
      if(fabs(lik[i] - lik0[i]) > *eps) 
        EM[nEM++] = i;
  }

  /* expand alpha to m x n */
  for(i = 0; i < *m; i++) {
    ll = 0;
    for(j = 0; j < *N; j++)
      atmp[j] = alpha[i][j];
    for(j = 0; j < *N; j++)
      for(k = 0; k < nGrp[j]; k++, ll++)
        alpha[i][idx[ll]] = atmp[j];
  }

}

/* R wrapper for calling EMFit directly */
void EMFitR(double *datR, double *HR, int *ind, int *idx, double *eps, 
  double *fitR, double *alphaR, 
  long *m, long *n, long *N, int *nGrp, long *iter) {

  long i, j, *EM;
  double **dat, **H, **fit, **alpha, **datC, *gm, **A0, **AA0, **AA, 
        *lik0, *lik, *tau, *sig, *atmp;

  /* allocate memory */
  dat = matrix(0, *m - 1, 0, *n - 1);
  H = matrix(0, *n - 1, 0, *n - 1);
  fit = matrix(0, *m - 1, 0, *n - 1);
  alpha = matrix(0, *m - 1, 0, *n - 1);
  datC = matrix(0, *m - 1, 0, *n - 1);
  gm = vector(0, *m - 1);
  A0 = matrix(0, *m - 1, 0, *N - 1);
  AA0 = matrix(0, *m - 1, 0, *N - 1);
  AA = matrix(0, *m - 1, 0, *N - 1);
  lik0 = vector(0, *m - 1);
  lik = vector(0, *m - 1);
  tau = vector(0, *m - 1);
  sig = vector(0, *m - 1);
  EM = ivector(0, *m - 1);
  atmp = vector(0, *N - 1);

  /* reform matrix inputs */
  for(i = 0; i < *n; i++) {
    for(j = 0; j < *m; j++) 
      dat[j][i] = datR[j + i * *m];

    for(j = 0; j < *n; j++) 
      H[j][i] = HR[j + i * *n];
  }

  /* call EM algorithm */
  EMFit(dat, H, ind, idx, eps, fit, alpha, datC, gm, A0, AA0, AA, lik0, 
    lik, tau, sig, EM, atmp, m, n, N, nGrp, iter, NULL, NULL, NULL, NULL);

  /* outputs */
  for(i = 0; i < *m; i++) {
    for(j = 0; j < *n; j++) {
      fitR[i + j * *m] = fit[i][j];
      alphaR[i + j * *m] = alpha[i][j];
    }
  }

  /* free memory */
  free_matrix(dat, 0, *m - 1, 0, *n - 1);
  free_matrix(H, 0, *n - 1, 0, *n - 1);
  free_matrix(fit, 0, *m - 1, 0, *n - 1);
  free_matrix(alpha, 0, *m - 1, 0, *n - 1);
  free_matrix(datC, 0, *m - 1, 0, *n - 1);
  free_vector(gm, 0,  - 1);
  free_matrix(A0, 0, *m - 1, 0, *N - 1);
  free_matrix(AA0, 0, *m - 1, 0, *N - 1);
  free_matrix(AA, 0, *m - 1, 0, *N - 1);
  free_vector(lik0, 0,  - 1);
  free_vector(lik, 0,  - 1);
  free_vector(tau, 0,  - 1);
  free_vector(sig, 0,  - 1);
  free_ivector(EM, 0, *m - 1);
  free_vector(atmp, 0,  - 1);
}

/***************************************************************************
utility functions:  adapted from Numerical Recipes in C
****************************************************************************/

/* quicksort routine */
void sortQK(long low, long high, long n, double *w) {
  if(low < high) {
    long lo = low, hi = high + 1;
    double elem = w[low];
    for (;;) {
      while ((lo < n) && (w[++lo] < elem));
      while ((hi >= 0) && (w[--hi] > elem));
      if (lo < hi) swapQK(lo, hi, w);
      else break;
    }

    swapQK(low, hi, w);
    sortQK(low, hi - 1, n, w);
    sortQK(hi + 1, high, n, w);
  }
}

/* swap function for use with sortQK() */
void swapQK(long i, long j, double *w) {
  double tmp = w[i];
  
  w[i] = w[j];
  w[j] = tmp;
}

/* allocate a int vector with subscript range v[nl...nh] */
int *intvector(long nl, long nh) {
  int *v;

  v = (int *) malloc((size_t)((nh - nl + 1 + NR_END) * sizeof(int)));
  if(!v) printf("\n allocation failure in intvector()\n");
  return v - nl + NR_END;
}

/* free a int vector allocated with ivector() */
void free_intvector(int *v, long nl, long nh) {
  free((FREE_ARG) (v + nl - NR_END));
}

/* allocate a long vector with subscript range v[nl...nh] */
long *ivector(long nl, long nh) {
  long *v;

  v = (long *) malloc((size_t)((nh - nl + 1 + NR_END) * sizeof(long)));
  if(!v) printf("\n allocation failure in ivector()\n");
  return v - nl + NR_END;
}

/* free a long vector allocated with ivector() */
void free_ivector(long *v, long nl, long nh) {
  free((FREE_ARG) (v + nl - NR_END));
}

/* allocate a long matrix with subscript ranges m[nrl...nrh][ncl...nch] */
long **imatrix(long nrl, long nrh, long ncl, long nch) {
  long i, nrow = nrh - nrl + 1, ncol = nch - ncl + 1;
  long **m;

  /* allocate pointers to rows */
  m = (long **) malloc((size_t)((nrow + NR_END) * sizeof(long*)));
  if(!m) printf("%s", "allocation fialure\n");

  m += NR_END;
  m -= nrl;

  /* set pointer to rows */
  m[nrl] = (long *) malloc((size_t)((nrow * ncol + NR_END) * sizeof(long)));
  if(!m[nrl]) printf("%s", "allocation fialure\n");
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for(i = nrl + 1; i <= nrh; i++) m[i] = m[i - 1] + ncol;
  return m;
}

/* free long matrix allocated with imatrix() */
void free_imatrix(long **m, long nrl, long nrh, long ncl, long nch) {
  free((FREE_ARG) (m[nrl] + ncl - NR_END));
  free((FREE_ARG) (m + nrl - NR_END));
}

/* allocate a double matrix with subscript ranges m[nrl...nrh][ncl...nch] */
double **matrix(long nrl, long nrh, long ncl, long nch) {
  long i, nrow = nrh - nrl + 1, ncol = nch - ncl + 1;
  double **m;

  /* allocate pointers to rows */
  m = (double **) malloc((size_t)((nrow + NR_END) * sizeof(double*)));
  if(!m) printf("%s", "allocation fialure\n");

  m += NR_END;
  m -= nrl;

  /* set pointer to rows */
  m[nrl] = (double *) malloc((size_t)((nrow * ncol + NR_END) * sizeof(double)));
  if(!m[nrl]) printf("%s", "allocation fialure\n");
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for(i = nrl + 1; i <= nrh; i++) m[i] = m[i - 1] + ncol;
  return m;
}

/* free double matrix allocated with matrix() */
void free_matrix(double **m, long nrl, long nrh, long ncl, long nch) {
  free((FREE_ARG) (m[nrl] + ncl - NR_END));
  free((FREE_ARG) (m + nrl - NR_END));
}

/* allocate a double vector with subscript range v[nl...nh] */
double *vector(long nl, long nh) {
  double *v;

  v = (double *) malloc((size_t) ((nh - nl + 1 + NR_END) * sizeof(double)));
  if(!v) printf("\n allocation failure in vector()\n");
  return v - nl + NR_END;
}

/* free double vector allocated with vector() */
void free_vector(double *v, long nl, long nh) {
  free((FREE_ARG) (v + nl - NR_END));
}
#ifdef __cplusplus
}
#endif
